﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ec2/EC2Request.h>
#include <aws/ec2/EC2_EXPORTS.h>

#include <utility>

namespace Aws {
namespace EC2 {
namespace Model {

/**
 */
class AssociateRouteTableRequest : public EC2Request {
 public:
  AWS_EC2_API AssociateRouteTableRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateRouteTable"; }

  AWS_EC2_API Aws::String SerializePayload() const override;

 protected:
  AWS_EC2_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The ID of the internet gateway or virtual private gateway.</p>
   */
  inline const Aws::String& GetGatewayId() const { return m_gatewayId; }
  inline bool GatewayIdHasBeenSet() const { return m_gatewayIdHasBeenSet; }
  template <typename GatewayIdT = Aws::String>
  void SetGatewayId(GatewayIdT&& value) {
    m_gatewayIdHasBeenSet = true;
    m_gatewayId = std::forward<GatewayIdT>(value);
  }
  template <typename GatewayIdT = Aws::String>
  AssociateRouteTableRequest& WithGatewayId(GatewayIdT&& value) {
    SetGatewayId(std::forward<GatewayIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a public IPv4 pool. A public IPv4 pool is a pool of IPv4 addresses
   * that you've brought to Amazon Web Services with BYOIP.</p>
   */
  inline const Aws::String& GetPublicIpv4Pool() const { return m_publicIpv4Pool; }
  inline bool PublicIpv4PoolHasBeenSet() const { return m_publicIpv4PoolHasBeenSet; }
  template <typename PublicIpv4PoolT = Aws::String>
  void SetPublicIpv4Pool(PublicIpv4PoolT&& value) {
    m_publicIpv4PoolHasBeenSet = true;
    m_publicIpv4Pool = std::forward<PublicIpv4PoolT>(value);
  }
  template <typename PublicIpv4PoolT = Aws::String>
  AssociateRouteTableRequest& WithPublicIpv4Pool(PublicIpv4PoolT&& value) {
    SetPublicIpv4Pool(std::forward<PublicIpv4PoolT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Checks whether you have the required permissions for the action, without
   * actually making the request, and provides an error response. If you have the
   * required permissions, the error response is <code>DryRunOperation</code>.
   * Otherwise, it is <code>UnauthorizedOperation</code>.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline AssociateRouteTableRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the subnet.</p>
   */
  inline const Aws::String& GetSubnetId() const { return m_subnetId; }
  inline bool SubnetIdHasBeenSet() const { return m_subnetIdHasBeenSet; }
  template <typename SubnetIdT = Aws::String>
  void SetSubnetId(SubnetIdT&& value) {
    m_subnetIdHasBeenSet = true;
    m_subnetId = std::forward<SubnetIdT>(value);
  }
  template <typename SubnetIdT = Aws::String>
  AssociateRouteTableRequest& WithSubnetId(SubnetIdT&& value) {
    SetSubnetId(std::forward<SubnetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the route table.</p>
   */
  inline const Aws::String& GetRouteTableId() const { return m_routeTableId; }
  inline bool RouteTableIdHasBeenSet() const { return m_routeTableIdHasBeenSet; }
  template <typename RouteTableIdT = Aws::String>
  void SetRouteTableId(RouteTableIdT&& value) {
    m_routeTableIdHasBeenSet = true;
    m_routeTableId = std::forward<RouteTableIdT>(value);
  }
  template <typename RouteTableIdT = Aws::String>
  AssociateRouteTableRequest& WithRouteTableId(RouteTableIdT&& value) {
    SetRouteTableId(std::forward<RouteTableIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_gatewayId;
  bool m_gatewayIdHasBeenSet = false;

  Aws::String m_publicIpv4Pool;
  bool m_publicIpv4PoolHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_subnetId;
  bool m_subnetIdHasBeenSet = false;

  Aws::String m_routeTableId;
  bool m_routeTableIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
