﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/SSEType.h>
#include <aws/ec2/model/SnapshotState.h>
#include <aws/ec2/model/Tag.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>Information about a snapshot.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/SnapshotInfo">AWS
 * API Reference</a></p>
 */
class SnapshotInfo {
 public:
  AWS_EC2_API SnapshotInfo() = default;
  AWS_EC2_API SnapshotInfo(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API SnapshotInfo& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>Description specified by the CreateSnapshotRequest that has been applied to
   * all snapshots.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  SnapshotInfo& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags associated with this snapshot.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  SnapshotInfo& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  SnapshotInfo& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether the snapshot is encrypted.</p>
   */
  inline bool GetEncrypted() const { return m_encrypted; }
  inline bool EncryptedHasBeenSet() const { return m_encryptedHasBeenSet; }
  inline void SetEncrypted(bool value) {
    m_encryptedHasBeenSet = true;
    m_encrypted = value;
  }
  inline SnapshotInfo& WithEncrypted(bool value) {
    SetEncrypted(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Source volume from which this snapshot was created.</p>
   */
  inline const Aws::String& GetVolumeId() const { return m_volumeId; }
  inline bool VolumeIdHasBeenSet() const { return m_volumeIdHasBeenSet; }
  template <typename VolumeIdT = Aws::String>
  void SetVolumeId(VolumeIdT&& value) {
    m_volumeIdHasBeenSet = true;
    m_volumeId = std::forward<VolumeIdT>(value);
  }
  template <typename VolumeIdT = Aws::String>
  SnapshotInfo& WithVolumeId(VolumeIdT&& value) {
    SetVolumeId(std::forward<VolumeIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Current state of the snapshot.</p>
   */
  inline SnapshotState GetState() const { return m_state; }
  inline bool StateHasBeenSet() const { return m_stateHasBeenSet; }
  inline void SetState(SnapshotState value) {
    m_stateHasBeenSet = true;
    m_state = value;
  }
  inline SnapshotInfo& WithState(SnapshotState value) {
    SetState(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Size of the volume from which this snapshot was created.</p>
   */
  inline int GetVolumeSize() const { return m_volumeSize; }
  inline bool VolumeSizeHasBeenSet() const { return m_volumeSizeHasBeenSet; }
  inline void SetVolumeSize(int value) {
    m_volumeSizeHasBeenSet = true;
    m_volumeSize = value;
  }
  inline SnapshotInfo& WithVolumeSize(int value) {
    SetVolumeSize(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Time this snapshot was started. This is the same for all snapshots initiated
   * by the same request.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  SnapshotInfo& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Progress this snapshot has made towards completing.</p>
   */
  inline const Aws::String& GetProgress() const { return m_progress; }
  inline bool ProgressHasBeenSet() const { return m_progressHasBeenSet; }
  template <typename ProgressT = Aws::String>
  void SetProgress(ProgressT&& value) {
    m_progressHasBeenSet = true;
    m_progress = std::forward<ProgressT>(value);
  }
  template <typename ProgressT = Aws::String>
  SnapshotInfo& WithProgress(ProgressT&& value) {
    SetProgress(std::forward<ProgressT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Account id used when creating this snapshot.</p>
   */
  inline const Aws::String& GetOwnerId() const { return m_ownerId; }
  inline bool OwnerIdHasBeenSet() const { return m_ownerIdHasBeenSet; }
  template <typename OwnerIdT = Aws::String>
  void SetOwnerId(OwnerIdT&& value) {
    m_ownerIdHasBeenSet = true;
    m_ownerId = std::forward<OwnerIdT>(value);
  }
  template <typename OwnerIdT = Aws::String>
  SnapshotInfo& WithOwnerId(OwnerIdT&& value) {
    SetOwnerId(std::forward<OwnerIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Snapshot id that can be used to describe this snapshot.</p>
   */
  inline const Aws::String& GetSnapshotId() const { return m_snapshotId; }
  inline bool SnapshotIdHasBeenSet() const { return m_snapshotIdHasBeenSet; }
  template <typename SnapshotIdT = Aws::String>
  void SetSnapshotId(SnapshotIdT&& value) {
    m_snapshotIdHasBeenSet = true;
    m_snapshotId = std::forward<SnapshotIdT>(value);
  }
  template <typename SnapshotIdT = Aws::String>
  SnapshotInfo& WithSnapshotId(SnapshotIdT&& value) {
    SetSnapshotId(std::forward<SnapshotIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the Outpost on which the snapshot is stored. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html">Amazon
   * EBS local snapshots on Outposts</a> in the <i>Amazon EBS User Guide</i>.</p>
   */
  inline const Aws::String& GetOutpostArn() const { return m_outpostArn; }
  inline bool OutpostArnHasBeenSet() const { return m_outpostArnHasBeenSet; }
  template <typename OutpostArnT = Aws::String>
  void SetOutpostArn(OutpostArnT&& value) {
    m_outpostArnHasBeenSet = true;
    m_outpostArn = std::forward<OutpostArnT>(value);
  }
  template <typename OutpostArnT = Aws::String>
  SnapshotInfo& WithOutpostArn(OutpostArnT&& value) {
    SetOutpostArn(std::forward<OutpostArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved for future use.</p>
   */
  inline SSEType GetSseType() const { return m_sseType; }
  inline bool SseTypeHasBeenSet() const { return m_sseTypeHasBeenSet; }
  inline void SetSseType(SSEType value) {
    m_sseTypeHasBeenSet = true;
    m_sseType = value;
  }
  inline SnapshotInfo& WithSseType(SSEType value) {
    SetSseType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Availability Zone or Local Zone of the snapshots. For example,
   * <code>us-west-1a</code> (Availability Zone) or <code>us-west-2-lax-1a</code>
   * (Local Zone).</p>
   */
  inline const Aws::String& GetAvailabilityZone() const { return m_availabilityZone; }
  inline bool AvailabilityZoneHasBeenSet() const { return m_availabilityZoneHasBeenSet; }
  template <typename AvailabilityZoneT = Aws::String>
  void SetAvailabilityZone(AvailabilityZoneT&& value) {
    m_availabilityZoneHasBeenSet = true;
    m_availabilityZone = std::forward<AvailabilityZoneT>(value);
  }
  template <typename AvailabilityZoneT = Aws::String>
  SnapshotInfo& WithAvailabilityZone(AvailabilityZoneT&& value) {
    SetAvailabilityZone(std::forward<AvailabilityZoneT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;

  bool m_encrypted{false};
  bool m_encryptedHasBeenSet = false;

  Aws::String m_volumeId;
  bool m_volumeIdHasBeenSet = false;

  SnapshotState m_state{SnapshotState::NOT_SET};
  bool m_stateHasBeenSet = false;

  int m_volumeSize{0};
  bool m_volumeSizeHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::String m_progress;
  bool m_progressHasBeenSet = false;

  Aws::String m_ownerId;
  bool m_ownerIdHasBeenSet = false;

  Aws::String m_snapshotId;
  bool m_snapshotIdHasBeenSet = false;

  Aws::String m_outpostArn;
  bool m_outpostArnHasBeenSet = false;

  SSEType m_sseType{SSEType::NOT_SET};
  bool m_sseTypeHasBeenSet = false;

  Aws::String m_availabilityZone;
  bool m_availabilityZoneHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
