/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.security;

import java.io.IOException;
import java.io.Serializable;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static org.hamcrest.CoreMatchers.instanceOf;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;

import org.springframework.util.SerializationUtils;

import org.apache.causeway.applib.services.iactnlayer.InteractionContext;
import org.apache.causeway.applib.services.user.UserMemento;

public abstract class EncodabilityContractTest {

    protected InteractionContext authContext;
    protected Serializable serializable;

    public EncodabilityContractTest() {
        super();
    }

    @BeforeEach
    public void setUp() throws Exception {
        serializable = createEncodable();
        authContext = InteractionContext.ofUserWithSystemDefaults(
                UserMemento.ofName("test")
                    .withAuthenticationCode("test_code"));
    }

    /**
     * Hook for subclasses to provide object to be tested.
     */
    protected abstract Serializable createEncodable();

    @Test
    public void shouldImplementEncodeable() throws Exception {
        assertThat(serializable, is(instanceOf(Serializable.class)));
    }

    @Test
    public void shouldRoundTrip() throws IOException, ClassNotFoundException {
        var decodedObject = SerializationUtils.clone(serializable);
        assertRoundtripped(decodedObject, serializable);
    }

    protected abstract void assertRoundtripped(Object decodedEncodable, Object originalEncodable);

}